#!/bin/bash

#
# Blackmagic Fusion Multi Debian package creator
# Release 2020-10-31
# By Daniel Tufvesson
#
MAKEFUSIONDEB_VERSION=1.0

check_command() {
    echo -n "Checking for ${1}..."
    if [[ ! -x "$(command -v $1)" ]]; then
	echo "Missing! Sorry, need '$1' to continue."
	exit 1
    fi
    echo "found!"
}

copy_object() {
    if [[ -e "$1" ]] ; then
	#echo "[COPY] $1 [TO] $2"
	if ! cp -rp "$1" "$2";
	then
	    ERRORS=$((ERRORS+1))
	fi
    else
	echo "[ERROR: MISSING] $1"
	ERRORS=$((ERRORS+1))
    fi
}

assert_object() {
    if ! [[ -e "$1" ]] ; then
	echo "[ERROR: MISSING] $1"
	ERRORS=$((ERRORS+1))
    fi
}

link_object() {
    #echo "[LINK] $2 [TO] $1"
    ln -s "$1" "$2"
}

create_directory() {
    if [[ -e "$1" ]] ; then
	echo "[ERROR: DIRECTORY EXISTS] $1"
	ERRORS=$((ERRORS+1))
    else
	#echo "[CREATE DIRECTORY] $1"
	mkdir -p "$1"
    fi
}

createf_directory() {
    if [[ -e "${1}" ]] ; then
	chmod a+w -R ${1}
	rm -rf ${1}
    fi
    create_directory "$1"
}

remove_directory() {
    if [[ -e "${1}" ]] ; then
	chmod a+w -R ${1}
	rm -rf ${1}
    else
	echo "[ERROR: MISSING] $1"
	ERRORS=$((ERRORS+1))
    fi
}

extract_tgz() {
    if [[ -f "$1" ]] ; then
	#echo "[EXTRACT] $1 [TO] $2/$3"
	tar -zxf "$1" -C "$2" "$3"
    else
	echo "[ERROR: MISSING] $1"
	ERRORS=$((ERRORS+1))
    fi
}

make_executable() {
    if [[ -e "$1" ]] ; then
	#echo "[MAKE EXECUTABLE] $1"
	chmod a+x "$1"
    else
	echo "[ERROR: MISSING] $1"
	ERRORS=$((ERRORS+1))
    fi
}

init_deb() {
    create_directory ${DEB_DIR}/DEBIAN
    cat > ${DEB_DIR}/DEBIAN/control <<EOF
Package: ${DEB_NAME}
Version: ${DEB_VERSION}
Architecture: amd64
Maintainer: Blackmagic Design
Description: ${FUSION_NAME} made from ${INSTALLER_ARCHIVE} using MakeFusionDeb ${MAKEFUSIONDEB_VERSION}
Conflicts: ${DEB_CONFLICTS}
EOF
    create_directory ${DEB_DIR}/usr/share/applications/
}

process9() {
    # Copy objects
    copy_object ${UNPACK_DIR}/Fusion ${FUSION_BASE_DIR}
    copy_object ${UNPACK_DIR}/FusionServer ${FUSION_BASE_DIR}
    copy_object ${UNPACK_DIR}/fuscript ${FUSION_BASE_DIR}
    copy_object ${UNPACK_DIR}/fusionscript.so ${FUSION_BASE_DIR}
    copy_object ${UNPACK_DIR}/libfusioncontrols.so ${FUSION_BASE_DIR}
    copy_object ${UNPACK_DIR}/libfusionformats.so ${FUSION_BASE_DIR}
    copy_object ${UNPACK_DIR}/libfusiongraphics.so ${FUSION_BASE_DIR}
    copy_object ${UNPACK_DIR}/libfusionoperators.so ${FUSION_BASE_DIR}
    copy_object ${UNPACK_DIR}/libfusionsystem.so ${FUSION_BASE_DIR}
    copy_object ${UNPACK_DIR}/BlackmagicFusion.py ${FUSION_BASE_DIR}
    copy_object ${UNPACK_DIR}/Brushes ${FUSION_BASE_DIR}
    copy_object ${UNPACK_DIR}/Filters ${FUSION_BASE_DIR}
    copy_object ${UNPACK_DIR}/Frameworks ${FUSION_BASE_DIR}
    copy_object ${UNPACK_DIR}/Fuses ${FUSION_BASE_DIR}
    copy_object ${UNPACK_DIR}/Guides ${FUSION_BASE_DIR}
    copy_object ${UNPACK_DIR}/Help ${FUSION_BASE_DIR}
    copy_object ${UNPACK_DIR}/Layouts ${FUSION_BASE_DIR}
    copy_object ${UNPACK_DIR}/Library ${FUSION_BASE_DIR}
    copy_object ${UNPACK_DIR}/LUTs ${FUSION_BASE_DIR}
    copy_object ${UNPACK_DIR}/Macros ${FUSION_BASE_DIR}
    copy_object ${UNPACK_DIR}/Plugins ${FUSION_BASE_DIR}
    copy_object ${UNPACK_DIR}/Scripts ${FUSION_BASE_DIR}
    copy_object ${UNPACK_DIR}/Settings ${FUSION_BASE_DIR}
    copy_object ${UNPACK_DIR}/Skins ${FUSION_BASE_DIR}
    copy_object ${UNPACK_DIR}/Templates.zip ${FUSION_VAR_DIR}
}

process16() {
    # Copy objects
    copy_object ${UNPACK_DIR}/Fusion ${FUSION_BASE_DIR}
    copy_object ${UNPACK_DIR}/FusionServer ${FUSION_BASE_DIR}
    copy_object ${UNPACK_DIR}/fuscript ${FUSION_BASE_DIR}
    copy_object ${UNPACK_DIR}/fusionscript.so ${FUSION_BASE_DIR}
    for FULIB in ${UNPACK_DIR}/lib*
    do
	copy_object ${FULIB} ${FUSION_BASE_DIR}
    done
    copy_object ${UNPACK_DIR}/BlackmagicFusion.py ${FUSION_BASE_DIR}
    copy_object ${UNPACK_DIR}/Frameworks ${FUSION_BASE_DIR}
    copy_object ${UNPACK_DIR}/Help ${FUSION_BASE_DIR}
    copy_object ${UNPACK_DIR}/Plugins ${FUSION_BASE_DIR}
    copy_object ${UNPACK_DIR}/Skins ${FUSION_BASE_DIR}

    copy_object ${UNPACK_DIR}/Templates.zip ${FUSION_VAR_DIR}
}

# Check argument
case $1 in
    *.run)
	INSTALLER_ARCHIVE=$1
	FILENAMEP1=$(echo $INSTALLER_ARCHIVE | cut -d'_' -f1 -)
	FILENAMEP2=$(echo $INSTALLER_ARCHIVE | cut -d'_' -f2 -)
	FILENAMEP3=$(echo $INSTALLER_ARCHIVE | cut -d'_' -f3 -)
	FILENAMEP4=$(echo $INSTALLER_ARCHIVE | cut -d'_' -f4 -)
	FILENAMEP5=$(echo $INSTALLER_ARCHIVE | cut -d'_' -f5 -)
	FILENAMEP6=$(echo $INSTALLER_ARCHIVE | cut -d'_' -f6 -)
	;;
    *)
	echo "Usage: $0 <Blackmagic_Fusion_*_Linux.run>"
	exit 1
	;;
esac

# Check if file exists
if [ ! -e $INSTALLER_ARCHIVE ];
then
    echo "[ERROR] File does not exist"
    exit 1
fi

# Validate and parse archive name
if [ "$FILENAMEP1" != "Blackmagic" ] || [ "$FILENAMEP2" != "Fusion" ];
then
    echo "ERROR: File does not look like a valid Blackmagic Fusion installer archive"
    exit 1
fi
if [ "$FILENAMEP6" == "installer.run" ] && [ "$FILENAMEP3" == "Studio" ];
then
    FUSION_NAME="Blackmagic Fusion Studio"
    FUSION_STUDIO=yes
    FUSION_VERSION=$FILENAMEP5
    DEB_NAME=blackmagic-fusion-studio
    DEB_CONFLICTS=blackmagic-fusion
elif [ "$FILENAMEP5" == "installer.run" ];
then
    FUSION_NAME="Blackmagic Fusion"
    FUSION_STUDIO=no
    FUSION_VERSION=$FILENAMEP4
    DEB_NAME=blackmagic-fusion
    DEB_CONFLICTS=blackmagic-fusion-studio
else
    echo "[ERROR] File does not look like a valid Blackmagic Fusion installer archive"
    exit 1
fi
UNPACK_DIR=./unpack-$DEB_NAME-$FUSION_VERSION
DEB_VERSION=${FUSION_VERSION}-mfd${MAKEFUSIONDEB_VERSION}
DEB_DIR=./${DEB_NAME}_${DEB_VERSION}_amd64
FUSION_BASE_DIR=${DEB_DIR}/opt/BlackmagicDesign/Fusion16
FUSION_VAR_DIR=${DEB_DIR}/var/BlackmagicDesign/Fusion
ERRORS=0

echo
echo "Detected ${FUSION_NAME} version ${FUSION_VERSION}"
echo

# Verify that we have the commands we need
check_command tar
check_command xorriso
check_command fakeroot
check_command dpkg-deb
echo

# Extract archive
echo "Extracting archive"
createf_directory ${UNPACK_DIR}
xorriso -osirrox on -indev ${INSTALLER_ARCHIVE} -extract / ${UNPACK_DIR}
if [[ -z "$(ls -A ${UNPACK_DIR})" ]];
then
    echo "[ERROR] Failed to extract Blackmagic Fusion archive"
    remove_directory ${UNPACK_DIR}
    exit 1
fi

# Create destination directories
createf_directory ${DEB_DIR}
createf_directory ${FUSION_BASE_DIR}
createf_directory ${FUSION_VAR_DIR}

# Initialize Debian package structure
init_deb

# Determine which conversion processs to use
case $FUSION_VERSION in
    9.*)
	echo "Using Fusion 9 conversion process"
	process9
	;;
    16.*)
	echo "Using Fusion 16 conversion process"
	process16
	;;
    *)
	echo "Unknown Fusion version"
	echo "Trying Fusion 16 conversion process"
	process16
	;;
esac

# Create udev rules
create_directory ${DEB_DIR}/lib/udev/rules.d
cat > ${DEB_DIR}/lib/udev/rules.d/76-sdx.rules <<EOF
SUBSYSTEM=="usb", ENV{DEVTYPE}=="usb_device", ATTRS{idVendor}=="096e", MODE="0666"
EOF

# Create desktop shortcut
assert_object ${DEB_DIR}/opt/BlackmagicDesign/Fusion16/Frameworks/icons/Fusion.png
cat > ${DEB_DIR}/usr/share/applications/${DEB_NAME}.desktop <<EOF
[Desktop Entry]
Version=1.0
Encoding=UTF-8
Path=/opt/BlackmagicDesign/Fusion16
Exec=/opt/BlackmagicDesign/Fusion16/Fusion %
Icon=/opt/BlackmagicDesign/Fusion16/Frameworks/icons/Fusion.png
Name=${FUSION_NAME}
Terminal=false
DBusActivatable=true
MimeType=application/x-fusioncomp;
Type=Application
StartupNotify=true
Categories=AudioVideo
Actions=Config;Manager;

[Desktop Action Config]
Exec=/opt/BlackmagicDesign/Fusion16/Fusion -config
Name=Configure Fusion 16

[Desktop Action Manager]
Exec=/opt/BlackmagicDesign/Fusion16/Fusion -manager
Name=Fusion 16 Render Manager
EOF

# Create Debian package
fakeroot dpkg-deb -b ${DEB_DIR} ${DEB_DIR}.deb
if [ $? -ne 0 ] ; then
    ERRORS=$((ERRORS+1))
fi

# Clean up
remove_directory ${UNPACK_DIR}
remove_directory ${DEB_DIR}

# Report
echo
echo "[DONE] errors reported ${ERRORS}"
echo

exit $ERRORS
