#!/bin/bash

#
# Blackmagic Fusion Multi Debian package creator
# Release 2024-08-29
# By Daniel Tufvesson
#
MAKEFUSIONDEB_VERSION=1.4.0

check_command() {
    echo -n "Checking for ${1}..."
    if [[ ! -x $(command -v "${1}") ]]; then
	echo "Missing! Sorry, need '$1' to continue."
	exit 1
    fi
    echo "found!"
}

copy_object() {
    if [[ -e "${1}" ]]; then
	if ! cp -rp "$1" "$2"; then
	    ERRORS=$((ERRORS+1))
	fi
    else
	echo "[ERROR: MISSING] $1"
	ERRORS=$((ERRORS+1))
    fi
}

assert_object() {
    if [[ ! -e "${1}" ]]; then
	echo "[ERROR: MISSING] $1"
	ERRORS=$((ERRORS+1))
    fi
}

link_object() {
    if [[ -e "${2}" ]]; then
	echo "[ERROR: LINK ALREADY EXISTS] $2"
	ERRORS=$((ERRORS+1))
    else
	if ! ln -s "${1}" "${2}"; then
	    echo "[ERROR: UNABLE TO CREATE LINK] $2 -> $1"
	    ERRORS=$((ERRORS+1))
	fi
    fi
}

create_directory() {
    if [[ -e "${1}" ]]; then
	echo "[ERROR: DIRECTORY ALREADY EXISTS] $1"
	ERRORS=$((ERRORS+1))
    else
	mkdir -p "${1}"
	chmod 755 "${1}"
    fi
    if [[ ! -w "${1}" ]]; then
	echo "[ERROR: DIRECTORY NOT WRITABLE] $1"
	ERRORS=$((ERRORS+1))
    fi
}

createf_directory() {
    if [[ -e "${1}" ]]; then
	chmod a+w -R "${1}"
	rm -rf "${1}"
    fi
    if [[ -e "${1}" ]]; then
	echo "[ERROR: UNABLE TO REPLACE DIRECTORY] $1"
	ERRORS=$((ERRORS+1))	
    fi
    create_directory "${1}"
}

remove_directory() {
    if [[ -e "${1}" ]]; then
	chmod a+w -R "${1}"
	rm -rf "${1}"
    else
	echo "[ERROR: MISSING] $1"
	ERRORS=$((ERRORS+1))
    fi
}

init_deb() {
    if [[ -z "$MAINTAINER" ]]; then
	MAINTAINER=$(whoami)@$(hostname)
    fi
    create_directory "${DEB_DIR}"/DEBIAN
    cat > "${DEB_DIR}"/DEBIAN/control <<EOF
Package: ${DEB_NAME}
Version: ${DEB_VERSION}
Priority: optional
Architecture: amd64
Maintainer: ${MAINTAINER}
Description: ${FUSION_NAME} made from ${INSTALLER_ARCHIVE} using MakeFusionDeb ${MAKEFUSIONDEB_VERSION}
Conflicts: ${DEB_CONFLICTS}
EOF
    create_directory "${DEB_DIR}"/usr/share/applications/
}

process9() {
    # Copy objects
    copy_object "${UNPACK_DIR}"/Fusion "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/FusionServer "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/fuscript "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/fusionscript.so "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/libfusioncontrols.so "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/libfusionformats.so "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/libfusiongraphics.so "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/libfusionoperators.so "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/libfusionsystem.so "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/BlackmagicFusion.py "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/Brushes "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/Filters "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/Frameworks "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/Fuses "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/Guides "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/Help "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/Layouts "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/Library "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/LUTs "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/Macros "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/Plugins "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/Scripts "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/Settings "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/Skins "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/Templates.zip "${FUSION_VAR_DIR}"
}

process161718() {
    # Copy objects
    copy_object "${UNPACK_DIR}"/Fusion "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/FusionServer "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/fuscript "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/fusionscript.so "${FUSION_BASE_DIR}"
    for FULIB in "${UNPACK_DIR}"/lib*
    do
	copy_object "${FULIB}" "${FUSION_BASE_DIR}"
    done
    copy_object "${UNPACK_DIR}"/BlackmagicFusion.py "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/Frameworks "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/Help "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/Plugins "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/Skins "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/Templates.* "${FUSION_VAR_DIR}"
}

process19() {
    # Copy objects
    copy_object "${UNPACK_DIR}"/Fusion "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/FusionServer "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/fuscript "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/fusionscript.so "${FUSION_BASE_DIR}"
    for FULIB in "${UNPACK_DIR}"/lib*
    do
	copy_object "${FULIB}" "${FUSION_BASE_DIR}"
    done
    copy_object "${UNPACK_DIR}"/BlackmagicFusion.py "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/Brushes "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/Filters "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/Frameworks "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/Fuses "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/Guides "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/Help "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/Library "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/LUTs "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/Macros "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/Models "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/Plugins "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/Scripts "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/Settings "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/Skins "${FUSION_BASE_DIR}"
    copy_object "${UNPACK_DIR}"/Templates.* "${FUSION_VAR_DIR}"
}


installer_extract_xorriso() {
    # Extract archive
    check_command xorriso
    echo "Extracting archive"
    createf_directory "${UNPACK_DIR}"
    xorriso -osirrox on -indev "${INSTALLER_ARCHIVE}" -extract / "${UNPACK_DIR}"
    if [[ -z $(ls -A "${UNPACK_DIR}") ]];
    then
	echo "[ERROR] Failed to extract DaVinci Resolve installer archive"
	remove_directory "${UNPACK_DIR}"
	exit 1
    fi
}

installer_extract_native() {
    # Extract archive
    echo "Extracting archive"
    createf_directory "${UNPACK_DIR}"
    chmod a+x "${INSTALLER_ARCHIVE}"
    if [[ ! -x "${INSTALLER_ARCHIVE}" ]];
    then
	echo "[ERROR] DaVinci Resolve installer archive is not executable"
	exit 1
    fi
    link_object "${UNPACK_DIR}" ./squashfs-root
    EXTRACT_OUT=$(./"${INSTALLER_ARCHIVE}" --appimage-extract 2>&1)
    rm -f ./squashfs-root
    find "${UNPACK_DIR}" -exec chmod a+r {} \;
    find "${UNPACK_DIR}" -type d -exec chmod a+x {} \;
    NUMEXTRACT=$(echo "${EXTRACT_OUT}" | grep "^squashfs-root/" -c)
    echo "Found ${NUMEXTRACT} objects"
    if [[ -z $(ls -A "${UNPACK_DIR}") ]];
    then
	echo "[ERROR] Failed to extract DaVinci Resolve installer archive"
	echo "${EXTRACT_OUT}" | tail
	remove_directory "${UNPACK_DIR}"
	exit 1
    fi
}

echo "MakeFusionDeb ${MAKEFUSIONDEB_VERSION}"

# Warn if running as root
if [[ $EUID -eq 0 ]];
then
    echo
    echo "[WARNING] This should not be run as root or using sudo"
    echo
fi

# Check argument
case $1 in
    *.run)
	INSTALLER_ARCHIVE=$1
	FILENAMEP1=$(echo "${INSTALLER_ARCHIVE}" | cut -d'_' -f1 -)
	FILENAMEP2=$(echo "${INSTALLER_ARCHIVE}" | cut -d'_' -f2 -)
	FILENAMEP3=$(echo "${INSTALLER_ARCHIVE}" | cut -d'_' -f3 -)
	FILENAMEP4=$(echo "${INSTALLER_ARCHIVE}" | cut -d'_' -f4 -)
	FILENAMEP5=$(echo "${INSTALLER_ARCHIVE}" | cut -d'_' -f5 -)
	FILENAMEP6=$(echo "${INSTALLER_ARCHIVE}" | cut -d'_' -f6 -)
	;;
    *)
	echo "Usage: $0 <Blackmagic_Fusion_*.run>"
	exit 1
	;;
esac

# Check if file exists
if [ ! -e "${INSTALLER_ARCHIVE}" ];
then
    echo "[ERROR] File does not exist"
    exit 1
fi

# Validate and parse archive name
if [ "$FILENAMEP1" != "Blackmagic" ] || [ "$FILENAMEP2" != "Fusion" ];
then
    echo "ERROR: File does not look like a valid Blackmagic Fusion installer archive"
    exit 1
fi
if [ "$FILENAMEP6" == "installer.run" ] && [ "$FILENAMEP3" == "Studio" ];
then
    FUSION_NAME="Blackmagic Fusion Studio"
    FUSION_VERSION=$FILENAMEP5
    DEB_NAME=blackmagic-fusion-studio
    DEB_CONFLICTS=blackmagic-fusion
elif [ "$FILENAMEP5" == "installer.run" ];
then
    FUSION_NAME="Blackmagic Fusion"
    FUSION_VERSION=$FILENAMEP4
    DEB_NAME=blackmagic-fusion
    DEB_CONFLICTS=blackmagic-fusion-studio
else
    echo "[ERROR] File does not look like a valid Blackmagic Fusion installer archive"
    exit 1
fi
UNPACK_DIR=./unpack-$DEB_NAME-$FUSION_VERSION
DEB_VERSION=${FUSION_VERSION}-mfd${MAKEFUSIONDEB_VERSION}
DEB_DIR=./${DEB_NAME}_${DEB_VERSION}_amd64
FUSION_BASE_DIR=${DEB_DIR}/opt/BlackmagicDesign/Fusion
FUSION_VAR_DIR=${DEB_DIR}/var/BlackmagicDesign/Fusion
ERRORS=0

echo
echo "Detected ${FUSION_NAME} version ${FUSION_VERSION}"
echo

# Verify that we have the commands we need
check_command tar
check_command fakeroot
check_command dpkg-deb
echo

# Create destination directories
createf_directory "${DEB_DIR}"
createf_directory "${FUSION_BASE_DIR}"
createf_directory "${FUSION_VAR_DIR}"

# Initialize Debian package structure
init_deb

# Determine which conversion processs to use
case $FUSION_VERSION in
    9.*)
	echo "Using Fusion 9 conversion process"
	installer_extract_xorriso
	process9
	;;
    16.*)
	echo "Using Fusion 16/17/18 conversion process"
	installer_extract_xorriso
	process161718
	;;
    17.*)
	echo "Using Fusion 16/17/18 conversion process"
	installer_extract_xorriso
	process161718
	;;
    18.*)
	echo "Using Fusion 16/17/18 conversion process"
	installer_extract_xorriso
	process161718
	;;
    19.*)
	echo "Using Fusion 19 conversion process"
	installer_extract_native
	process19
	;;
    *)
	echo "Unknown Fusion version"
	echo "Trying Fusion 19 conversion process"
	installer_extract_native
	process19
	;;
esac

# Create udev rules
create_directory "${DEB_DIR}"/lib/udev/rules.d
cat > "${DEB_DIR}"/lib/udev/rules.d/76-sdx.rules <<EOF
SUBSYSTEM=="usb", ENV{DEVTYPE}=="usb_device", ATTRS{idVendor}=="096e", MODE="0666"
EOF

# Create desktop shortcut
assert_object "${DEB_DIR}"/opt/BlackmagicDesign/Fusion/Frameworks/icons/Fusion.png
cat > "${DEB_DIR}"/usr/share/applications/"${DEB_NAME}".desktop <<EOF
[Desktop Entry]
Version=1.0
Encoding=UTF-8
Path=/opt/BlackmagicDesign/Fusion
Exec=/opt/BlackmagicDesign/Fusion/Fusion %
Icon=/opt/BlackmagicDesign/Fusion/Frameworks/icons/Fusion.png
Name=${FUSION_NAME}
Terminal=false
DBusActivatable=true
MimeType=application/x-fusioncomp;
Type=Application
StartupNotify=true
Categories=AudioVideo
Actions=Config;Manager;

[Desktop Action Config]
Exec=/opt/BlackmagicDesign/Fusion/Fusion -config
Name=Configure Fusion 16

[Desktop Action Manager]
Exec=/opt/BlackmagicDesign/Fusion/Fusion -manager
Name=Fusion 16 Render Manager
EOF

if [[ -z "$CI_TEST" ]]; then

    # Create Debian package
    echo "Creating Debian package (This can take a while. Do not interrupt)"
    if ! fakeroot dpkg-deb -b "${DEB_DIR}" "${DEB_DIR}".deb;
    then
	ERRORS=$((ERRORS+1))
    fi
    
    # Clean up
    remove_directory "${UNPACK_DIR}"
    remove_directory "${DEB_DIR}"
fi

# Report
echo
echo "[DONE] errors reported ${ERRORS}"
echo

exit $ERRORS
