#!/bin/sh

#
# Fusion 16.2.3-1 Debian package creator
# Release 2020-07-05
# By Daniel Tufvesson
#

check_command() {
    if [ ! -x "$(command -v $1)" ]; then
	echo "Sorry. Need '$1' to continue."
	exit 1
    fi
}

copy_object() {
    if [ -e "$1" ] ; then
	echo "[COPY] $1 [TO] $2"
	cp -rp "$1" "$2"
	if [ $? -ne 0 ] ; then
	    ERRORS=$((ERRORS+1))
	fi
    else
	echo "[MISSING] $1"
	ERRORS=$((ERRORS+1))
    fi
}

create_directory() {
    if [ -e "$1" ] ; then
	echo "[DIRECTORY EXISTS] $1"
	ERRORS=$((ERRORS+1))
    else
	echo "[CREATE DIRECTORY] $1"
	mkdir -p "$1"
    fi
}

make_executable() {
    if [ -e "$1" ] ; then
	echo "[MAKE EXECUTABLE] $1"
	chmod a+x "$1"
    else
	echo "[MISSING] $1"
	ERRORS=$((ERRORS+1))
    fi
}

# Setup variables
FUSION_VERSION=16.2.3
DEB_STEPPING=1
case $1 in
    studio)
	FUSION_NAME="Blackmagic Fusion Studio"
	INSTALLER_ARCHIVE=Blackmagic_Fusion_Studio_Linux_${FUSION_VERSION}_installer.run
	DEB_NAME=blackmagic-fusion-studio
	DEB_CONFLICTS=blackmagic-fusion
	UNPACK_DIR=./unpack-Blackmagic_Fusion_StudioLinux_${FUSION_VERSION}_installer
	;;
    lite)
	FUSION_NAME="Blackmagic Fusion"
	INSTALLER_ARCHIVE=Blackmagic_Fusion_Linux_${FUSION_VERSION}_installer.run
	DEB_NAME=blackmagic-fusion
	DEB_CONFLICTS=blackmagic-fusion-studio
	UNPACK_DIR=./unpack-Blackmagic_Fusion_Linux_${FUSION_VERSION}_installer
	;;
    *)
	echo "\n Usage: $0 <studio|lite>\n"
	exit 1
	;;
esac
DEB_VERSION=${FUSION_VERSION}-${DEB_STEPPING}
DEB_DIR=./${DEB_NAME}_${DEB_VERSION}_amd64
FUSION_BASE_DIR=${DEB_DIR}/opt/BlackmagicDesign/Fusion16
FUSION_VAR_DIR=${DEB_DIR}/var/BlackmagicDesign/Fusion
ERRORS=0

echo "Preparing ${FUSION_NAME} version ${FUSION_VERSION}"

# Extract Fusion Archive
if [ ! -e "${INSTALLER_ARCHIVE}" ] ; then
    echo "Sorry. Need '${INSTALLER_ARCHIVE}' to continue. Download the archive from www.blackmagicdesign.com"
    exit 1
fi
create_directory ${UNPACK_DIR}
check_command xorriso
xorriso -osirrox on -indev ${INSTALLER_ARCHIVE} -extract / ${UNPACK_DIR}

# Create destination directories
create_directory ${DEB_DIR}
create_directory ${FUSION_BASE_DIR}
create_directory ${FUSION_VAR_DIR}

# Copy objects
copy_object ${UNPACK_DIR}/Fusion ${FUSION_BASE_DIR}
copy_object ${UNPACK_DIR}/FusionServer ${FUSION_BASE_DIR}
copy_object ${UNPACK_DIR}/fuscript ${FUSION_BASE_DIR}
copy_object ${UNPACK_DIR}/fusionscript.so ${FUSION_BASE_DIR}
copy_object ${UNPACK_DIR}/libcublas.so.10.0 ${FUSION_BASE_DIR}
copy_object ${UNPACK_DIR}/libfusioncontrols.so ${FUSION_BASE_DIR}
copy_object ${UNPACK_DIR}/libFusionCore.so ${FUSION_BASE_DIR}
copy_object ${UNPACK_DIR}/libfusionformats.so ${FUSION_BASE_DIR}
copy_object ${UNPACK_DIR}/libfusiongraphics.so ${FUSION_BASE_DIR}
copy_object ${UNPACK_DIR}/libfusionoperators.so ${FUSION_BASE_DIR}
copy_object ${UNPACK_DIR}/libfusionsystem.so ${FUSION_BASE_DIR}
copy_object ${UNPACK_DIR}/libgpudetect.so ${FUSION_BASE_DIR}
copy_object ${UNPACK_DIR}/libnvrtc-builtins.so ${FUSION_BASE_DIR}
copy_object ${UNPACK_DIR}/libnvrtc.so.10.0 ${FUSION_BASE_DIR}
copy_object ${UNPACK_DIR}/BlackmagicFusion.py ${FUSION_BASE_DIR}

copy_object ${UNPACK_DIR}/Frameworks ${FUSION_BASE_DIR}
copy_object ${UNPACK_DIR}/Help ${FUSION_BASE_DIR}
copy_object ${UNPACK_DIR}/Plugins ${FUSION_BASE_DIR}
copy_object ${UNPACK_DIR}/Skins ${FUSION_BASE_DIR}

copy_object ${UNPACK_DIR}/Templates.zip ${FUSION_VAR_DIR}

# Create udev rules
create_directory ${DEB_DIR}/lib/udev/rules.d
cat > ${DEB_DIR}/lib/udev/rules.d/76-sdx.rules <<EOF
SUBSYSTEM=="usb", ENV{DEVTYPE}=="usb_device", ATTRS{idVendor}=="096e", MODE="0666"
EOF

# Create desktop shortcut
create_directory ${DEB_DIR}/usr/share/applications/
cat > ${DEB_DIR}/usr/share/applications/${DEB_NAME}.desktop <<EOF
[Desktop Entry]
Version=1.0
Encoding=UTF-8
Path=/opt/BlackmagicDesign/Fusion16
Exec=/opt/BlackmagicDesign/Fusion16/Fusion %
Icon=/opt/BlackmagicDesign/Fusion16/Frameworks/icons/Fusion.png
Name=${FUSION_NAME}
Terminal=false
DBusActivatable=true
MimeType=application/x-fusioncomp;
Type=Application
StartupNotify=true
Categories=AudioVideo
Actions=Config;Manager;

[Desktop Action Config]
Exec=/opt/BlackmagicDesign/Fusion16/Fusion -config
Name=Configure Fusion 16

[Desktop Action Manager]
Exec=/opt/BlackmagicDesign/Fusion16/Fusion -manager
Name=Fusion 16 Render Manager
EOF

# Create Debian package
create_directory ${DEB_DIR}/DEBIAN
cat > ${DEB_DIR}/DEBIAN/control <<EOF
Package: ${DEB_NAME}
Version: ${DEB_VERSION}
Architecture: amd64
Maintainer: Blackmagic Design
Description: ${FUSION_NAME} made from ${INSTALLER_ARCHIVE}
Conflicts: ${DEB_CONFLICTS}
EOF
check_command fakeroot
check_command dpkg-deb
fakeroot dpkg-deb -b ${DEB_DIR} ${DEB_DIR}.deb
if [ $? -ne 0 ] ; then
    ERRORS=$((ERRORS+1))
fi

# Clean up
rm -rf ${DEB_DIR}
rm -rf ${UNPACK_DIR}

# Report
echo "[DONE] errors reported ${ERRORS}"

exit $ERRORS
