#!/bin/sh

#
# DaVinci Resolve 15.1 Debian package creator
# Release 2018-09-14
# By Daniel Tufvesson
#

check_command() {
    if [ ! -x "$(command -v $1)" ]; then
	echo "Sorry. Need '$1' to continue."
	exit 1
    fi
}

copy_object() {
    if [ -e "$1" ] ; then
	echo "[COPY] $1 [TO] $2"
	cp -rp "$1" "$2"
	if [ $? -ne 0 ] ; then
	    ERRORS=$((ERRORS+1))
	fi
    else
	echo "[MISSING] $1"
	ERRORS=$((ERRORS+1))
    fi
}

create_directory() {
    if [ -e "$1" ] ; then
	echo "[DIRECTORY EXISTS] $1"
	ERRORS=$((ERRORS+1))
    else
	echo "[CREATE DIRECTORY] $1"
	mkdir -p "$1"
    fi
}

make_executable() {
    if [ -e "$1" ] ; then
	echo "[MAKE EXECUTABLE] $1"
	chmod a+x "$1"
    else
	echo "[MISSING] $1"
	ERRORS=$((ERRORS+1))
    fi
}

extract_inplace_gz() {
    check_command gunzip
    if [ -f "$1" ] ; then
	echo "[EXTRACT] $1"
	gunzip -f "$1"
	if [ -e "$1" ] ; then
	    rm "$1"
	fi
    else
	echo "[MISSING] $1"
	ERRORS=$((ERRORS+1))
    fi
}

extract_inplace_tgz() {
    check_command tar
    if [ -f "$1" ] ; then
	echo "[EXTRACT] $1"
	tar zxf "$1" -C "$(dirname $1)"
	if [ -e "$1" ] ; then
	    rm "$1"
	fi
    else
	echo "[MISSING] $1"
	ERRORS=$((ERRORS+1))
    fi
}

extract_tgz() {
    check_command tar
    if [ -f "$1" ] ; then
	echo "[EXTRACT] $1 [TO] $2"
	tar zxf "$1" -C "$2"
    else
	echo "[MISSING] $1"
	ERRORS=$((ERRORS+1))
    fi
}

extract_zip() {
    check_command unzip
    if [ -f "$1" ] ; then
	echo "[EXTRACT] $1 [TO] $2"
	unzip -qq "$1" -d "$2"
    else
	echo "[MISSING] $1"
	ERRORS=$((ERRORS+1))
    fi
}

# Setup variables
RESOLVE_VERSION=15.1
DEB_STEPPING=1
case $1 in
    studio)
	RESOLVE_NAME="DaVinci Resolve Studio"
	INSTALLER_ARCHIVE=DaVinci_Resolve_Studio_${RESOLVE_VERSION}_Linux.sh
	DEB_NAME=davinci-resolve-studio
	DEB_CONFLICTS=davinci-resolve
	UNPACK_DIR=./unpack-DaVinci_Resolve_Studio_${RESOLVE_VERSION}_Linux
	;;
    lite)
	RESOLVE_NAME="DaVinci Resolve"
	INSTALLER_ARCHIVE=DaVinci_Resolve_${RESOLVE_VERSION}_Linux.sh
	DEB_NAME=davinci-resolve
	DEB_CONFLICTS=davinci-resolve-studio
	UNPACK_DIR=./unpack-DaVinci_Resolve_${RESOLVE_VERSION}_Linux
	;;
    *)
	echo "\n Usage: $0 <studio|lite>\n"
	exit 1
	;;
esac
DEB_VERSION=${RESOLVE_VERSION}-${DEB_STEPPING}
DEB_DIR=./${DEB_NAME}_${DEB_VERSION}_amd64
RESOLVE_BASE_DIR=${DEB_DIR}/opt/resolve
ERRORS=0

echo "Preparing ${RESOLVE_NAME} version ${RESOLVE_VERSION}"

# Extract DaVinci Resolve Archive
if [ ! -e "${INSTALLER_ARCHIVE}" ] ; then
    echo "Sorry. Need '${INSTALLER_ARCHIVE}' to continue. Download the archive from www.blackmagicdesign.com"
    exit 1
fi
create_directory ${UNPACK_DIR}
sh ${INSTALLER_ARCHIVE} --noexec --keep --target ${UNPACK_DIR}

# Create destination directories
create_directory ${DEB_DIR}
create_directory ${RESOLVE_BASE_DIR}

# Create directories
create_directory ${RESOLVE_BASE_DIR}/bin
create_directory ${RESOLVE_BASE_DIR}/configs
create_directory ${RESOLVE_BASE_DIR}/docs
create_directory ${RESOLVE_BASE_DIR}/easyDCP
create_directory ${RESOLVE_BASE_DIR}/graphics
create_directory ${RESOLVE_BASE_DIR}/logs
create_directory ${RESOLVE_BASE_DIR}/scripts
create_directory ${RESOLVE_BASE_DIR}/.LUT
create_directory ${RESOLVE_BASE_DIR}/.license
create_directory ${RESOLVE_BASE_DIR}/.crashreport
create_directory ${RESOLVE_BASE_DIR}/DolbyVision
create_directory ${RESOLVE_BASE_DIR}/Fairlight
create_directory ${RESOLVE_BASE_DIR}/Media
create_directory ${RESOLVE_BASE_DIR}/"Resolve Disk Database"

# Copy objects
copy_object ${UNPACK_DIR}/UI_Resource ${RESOLVE_BASE_DIR}
copy_object ${UNPACK_DIR}/libs ${RESOLVE_BASE_DIR}
copy_object ${UNPACK_DIR}/plugins ${RESOLVE_BASE_DIR}
copy_object ${UNPACK_DIR}/Developer ${RESOLVE_BASE_DIR}
copy_object ${UNPACK_DIR}/LUT ${RESOLVE_BASE_DIR}
copy_object ${UNPACK_DIR}/panels/DaVinciPanelDaemon ${RESOLVE_BASE_DIR}/bin/

copy_object ${UNPACK_DIR}/DaVinci_Resolve_Manual.pdf ${RESOLVE_BASE_DIR}/docs/
copy_object ${UNPACK_DIR}/resolve ${RESOLVE_BASE_DIR}/bin/
copy_object ${UNPACK_DIR}/rsf/bmdpaneld ${RESOLVE_BASE_DIR}/bin/
copy_object ${UNPACK_DIR}/rsf/run_bmdpaneld ${RESOLVE_BASE_DIR}/bin/
copy_object ${UNPACK_DIR}/rsf/BMDPanelFirmware ${RESOLVE_BASE_DIR}/bin/
copy_object ${UNPACK_DIR}/rsf/DPDecoder ${RESOLVE_BASE_DIR}/bin/
copy_object ${UNPACK_DIR}/rsf/qt.conf ${RESOLVE_BASE_DIR}/bin/
copy_object ${UNPACK_DIR}/rsf/ShowDpxHeader ${RESOLVE_BASE_DIR}/bin/
copy_object ${UNPACK_DIR}/rsf/TestIO ${RESOLVE_BASE_DIR}/bin/
copy_object ${UNPACK_DIR}/rsf/deviceQuery ${RESOLVE_BASE_DIR}/bin/
copy_object ${UNPACK_DIR}/rsf/bandwidthTest ${RESOLVE_BASE_DIR}/bin/
copy_object ${UNPACK_DIR}/rsf/oclDeviceQuery ${RESOLVE_BASE_DIR}/bin/
copy_object ${UNPACK_DIR}/rsf/oclBandwidthTest ${RESOLVE_BASE_DIR}/bin/
copy_object ${UNPACK_DIR}/rsf/VstScanner ${RESOLVE_BASE_DIR}/bin/
copy_object ${UNPACK_DIR}/rsf/script.getlogs.v4 ${RESOLVE_BASE_DIR}/scripts/
copy_object ${UNPACK_DIR}/Onboarding ${RESOLVE_BASE_DIR}
copy_object ${UNPACK_DIR}/rsf/script.start ${RESOLVE_BASE_DIR}/scripts/
copy_object ${UNPACK_DIR}/rsf/Control ${RESOLVE_BASE_DIR}
copy_object ${UNPACK_DIR}/rsf/DV_Resolve.png ${RESOLVE_BASE_DIR}/graphics
copy_object ${UNPACK_DIR}/rsf/default-config-linux.dat ${RESOLVE_BASE_DIR}/configs/config.dat-pkg-default
copy_object ${UNPACK_DIR}/rsf/log-conf.xml ${RESOLVE_BASE_DIR}/configs/log-conf.xml-pkg-default
copy_object ${UNPACK_DIR}/rsf/default_cm_config.bin ${RESOLVE_BASE_DIR}/DolbyVision/config.bin-pkg-default

# Extract archives
extract_tgz ${UNPACK_DIR}/panels/libusb-1.0.tgz ${RESOLVE_BASE_DIR}/bin
extract_tgz ${UNPACK_DIR}/panels/dvpanel-framework-linux-x86_64.tgz ${RESOLVE_BASE_DIR}/bin
extract_tgz ${UNPACK_DIR}/panels/dvpanel-utility-linux-x86_64.tgz ${RESOLVE_BASE_DIR}
extract_zip ${UNPACK_DIR}/rsf/fusion_presets.zip ${RESOLVE_BASE_DIR}
extract_inplace_gz ${RESOLVE_BASE_DIR}/LUT/trim_lut0.dpx.gz
extract_inplace_tgz ${RESOLVE_BASE_DIR}/libs/aaf_runtime_libs_x86_64.tgz
extract_inplace_tgz ${RESOLVE_BASE_DIR}/libs/easydcp-rt-linux-x86_64.tgz
extract_inplace_tgz ${RESOLVE_BASE_DIR}/libs/fairlight-lib-linux-x86_64.tgz
extract_inplace_tgz ${RESOLVE_BASE_DIR}/libs/fusion-rt-linux-x86_64.tgz
extract_inplace_tgz ${RESOLVE_BASE_DIR}/libs/ffmpeg-rt-linux-x86_64.tgz
extract_inplace_tgz ${RESOLVE_BASE_DIR}/libs/graphviz-rt-linux-x86_64.tgz
extract_inplace_tgz ${RESOLVE_BASE_DIR}/libs/log4cxx_runtime_libs_x64.tgz
extract_inplace_tgz ${RESOLVE_BASE_DIR}/libs/sonyrawdev-rt-linux-x86_64.tgz
extract_inplace_tgz ${RESOLVE_BASE_DIR}/libs/smdk-rt-linux-x86_64.tgz
extract_inplace_tgz ${RESOLVE_BASE_DIR}/libs/sstp-rt-linux-x86_64.tgz
extract_inplace_tgz ${RESOLVE_BASE_DIR}/libs/sxavc-rt-linux-x86_64.tgz

# Create udev rules
create_directory ${DEB_DIR}/lib/udev/rules.d
cat > ${DEB_DIR}/lib/udev/rules.d/75-davincipanel.rules <<EOF
SUBSYSTEM=="usb", ENV{DEVTYPE}=="usb_device", ATTRS{idVendor}=="1edb", MODE="0666"
EOF
cat > ${DEB_DIR}/lib/udev/rules.d/75-sdx.rules <<EOF
SUBSYSTEM=="usb", ENV{DEVTYPE}=="usb_device", ATTRS{idVendor}=="096e", MODE="0666"
EOF

# Create desktop shortcut
create_directory ${DEB_DIR}/usr/share/applications/
cat > ${DEB_DIR}/usr/share/applications/${DEB_NAME}.desktop <<EOF
[Desktop Entry]
Version=${DEB_VERSION}
Encoding=UTF-8
Exec=/opt/resolve/bin/resolve
Icon=/opt/resolve/graphics/DV_Resolve.png
MimeType=
Name=${RESOLVE_NAME}
Terminal=false
Type=Application
Categories=AudioVideo
EOF

# Create Debian package
create_directory ${DEB_DIR}/DEBIAN
cat > ${DEB_DIR}/DEBIAN/control <<EOF
Package: ${DEB_NAME}
Version: ${DEB_VERSION}
Architecture: amd64
Maintainer: Blackmagic Design
Description: ${RESOLVE_NAME} made from ${INSTALLER_ARCHIVE}
Conflicts: ${DEB_CONFLICTS}
EOF
cat > ${DEB_DIR}/DEBIAN/postinst <<EOF
#!/bin/sh
test -e /usr/lib/x86_64-linux-gnu/libcrypto.so.1.0.0 && ln -sf /usr/lib/x86_64-linux-gnu/libcrypto.so.1.0.0 /opt/resolve/libs/libcrypto.so.10
test -e /lib/x86_64-linux-gnu/libcrypto.so.1.0.0 && ln -sf /lib/x86_64-linux-gnu/libcrypto.so.1.0.0 /opt/resolve/libs/libcrypto.so.10
test -e /usr/lib/x86_64-linux-gnu/libssl.so.1.0.0 && ln -sf /usr/lib/x86_64-linux-gnu/libssl.so.1.0.0 /opt/resolve/libs/libssl.so.10
test -e /lib/x86_64-linux-gnu/libssl.so.1.0.0 && ln -sf /lib/x86_64-linux-gnu/libssl.so.1.0.0 /opt/resolve/libs/libssl.so.10
test -e /opt/resolve/configs/config.dat || cp /opt/resolve/configs/config.dat-pkg-default /opt/resolve/configs/config.dat
test -e /opt/resolve/configs/log-conf.xml || cp /opt/resolve/configs/log-conf.xml-pkg-default /opt/resolve/configs/log-conf.xml
test -e /opt/resolve/DolbyVision/config.bin || cp /opt/resolve/DolbyVision/config.bin-pkg-default /opt/resolve/DolbyVision/config.bin
chmod -R a+rw /opt/resolve/configs
chmod -R a+rw /opt/resolve/Media
chmod -R a+rw /opt/resolve/DolbyVision
chmod -R a+rw /opt/resolve/Fairlight
chmod -R a+rw /opt/resolve/easyDCP
chmod -R a+rw /opt/resolve/Developer
chmod -R a+rw /opt/resolve/"Resolve Disk Database"
chmod -R a+rw /opt/resolve/LUT
chmod -R a+rw /opt/resolve/.LUT
chmod -R a+rw /opt/resolve/.license
chmod -R a+rw /opt/resolve/.crashreport
chmod -R a+rw /opt/resolve/logs
EOF
make_executable ${DEB_DIR}/DEBIAN/postinst
cat > ${DEB_DIR}/DEBIAN/prerm <<EOF
#!/bin/sh
rm -f /opt/resolve/libs/libcrypto.so.10
rm -f /opt/resolve/libs/libssl.so.10
EOF
make_executable ${DEB_DIR}/DEBIAN/prerm
check_command fakeroot
check_command dpkg-deb
fakeroot dpkg-deb -b ${DEB_DIR} ${DEB_DIR}.deb
if [ $? -ne 0 ] ; then
    ERRORS=$((ERRORS+1))
fi

# Clean up
rm -rf ${DEB_DIR}
rm -rf ${UNPACK_DIR}

# Report
echo "[DONE] errors reported ${ERRORS}"

exit $ERRORS
