#!/bin/sh

#
# DaVinci Resolve 16.0b1 Debian package creator
# Release 2019-04-13
# By Daniel Tufvesson
#

check_command() {
    if [ ! -x "$(command -v $1)" ]; then
	echo "Sorry. Need '$1' to continue."
	exit 1
    fi
}

copy_object() {
    if [ -e "$1" ] ; then
	echo "[COPY] $1 [TO] $2"
	cp -rp "$1" "$2"
	if [ $? -ne 0 ] ; then
	    ERRORS=$((ERRORS+1))
	fi
    else
	echo "[ERROR: MISSING] $1"
	ERRORS=$((ERRORS+1))
    fi
}

create_directory() {
    if [ -e "$1" ] ; then
	echo "[ERROR: DIRECTORY EXISTS] $1"
	ERRORS=$((ERRORS+1))
    else
	echo "[CREATE DIRECTORY] $1"
	mkdir -p "$1"
    fi
}

extract_tgz() {
    check_command tar
    if [ -f "$1" ] ; then
	echo "[EXTRACT] $1 [TO] $2"
	tar -zxf "$1" -C "$2" $3
    else
	echo "[ERROR: MISSING] $1"
    fi
}

make_executable() {
    if [ -e "$1" ] ; then
	echo "[MAKE EXECUTABLE] $1"
	chmod a+x "$1"
    else
	echo "[ERROR: MISSING] $1"
	ERRORS=$((ERRORS+1))
    fi
}

# Setup variables
RESOLVE_VERSION=16.0b1
DEB_STEPPING=1
case $1 in
    studio)
	RESOLVE_NAME="DaVinci Resolve Studio"
	INSTALLER_ARCHIVE=DaVinci_Resolve_Studio_${RESOLVE_VERSION}_Linux.run
	DEB_NAME=davinci-resolve-studio
	DEB_CONFLICTS=davinci-resolve
	UNPACK_DIR=./unpack-DaVinci_Resolve_Studio_${RESOLVE_VERSION}_Linux
	;;
    lite)
	RESOLVE_NAME="DaVinci Resolve"
	INSTALLER_ARCHIVE=DaVinci_Resolve_${RESOLVE_VERSION}_Linux.run
	DEB_NAME=davinci-resolve
	DEB_CONFLICTS=davinci-resolve-studio
	UNPACK_DIR=./unpack-DaVinci_Resolve_${RESOLVE_VERSION}_Linux
	;;
    *)
	echo "\n Usage: $0 <studio|lite>\n"
	exit 1
	;;
esac
DEB_VERSION=${RESOLVE_VERSION}-${DEB_STEPPING}
DEB_DIR=./${DEB_NAME}_${DEB_VERSION}_amd64
RESOLVE_BASE_DIR=${DEB_DIR}/opt/resolve
ERRORS=0

echo "Preparing ${RESOLVE_NAME} version ${RESOLVE_VERSION}"

# Extract DaVinci Resolve Archive
if [ ! -e "${INSTALLER_ARCHIVE}" ] ; then
    echo "Sorry. Need '${INSTALLER_ARCHIVE}' to continue. Download the archive from www.blackmagicdesign.com"
    exit 1
fi
create_directory ${UNPACK_DIR}
check_command xorriso
xorriso -osirrox on -indev ${INSTALLER_ARCHIVE} -extract / ${UNPACK_DIR}

# Create destination directories
create_directory ${DEB_DIR}
create_directory ${RESOLVE_BASE_DIR}

# Create directories
create_directory ${RESOLVE_BASE_DIR}/easyDCP
create_directory ${RESOLVE_BASE_DIR}/scripts
create_directory ${RESOLVE_BASE_DIR}/.license
create_directory ${RESOLVE_BASE_DIR}/share

# Copy objects
copy_object ${UNPACK_DIR}/bin ${RESOLVE_BASE_DIR}
copy_object ${UNPACK_DIR}/Control ${RESOLVE_BASE_DIR}
copy_object ${UNPACK_DIR}/DaVinci\ Resolve\ Panels\ Setup ${RESOLVE_BASE_DIR}
copy_object ${UNPACK_DIR}/Developer ${RESOLVE_BASE_DIR}
copy_object ${UNPACK_DIR}/docs ${RESOLVE_BASE_DIR}
copy_object ${UNPACK_DIR}/Fusion ${RESOLVE_BASE_DIR}
copy_object ${UNPACK_DIR}/graphics ${RESOLVE_BASE_DIR}
copy_object ${UNPACK_DIR}/libs ${RESOLVE_BASE_DIR}
copy_object ${UNPACK_DIR}/LUT ${RESOLVE_BASE_DIR}
copy_object ${UNPACK_DIR}/Onboarding ${RESOLVE_BASE_DIR}
copy_object ${UNPACK_DIR}/plugins ${RESOLVE_BASE_DIR}
copy_object ${UNPACK_DIR}/UI_Resource ${RESOLVE_BASE_DIR}
copy_object ${UNPACK_DIR}/scripts/script.checkfirmware ${RESOLVE_BASE_DIR}/scripts/
copy_object ${UNPACK_DIR}/scripts/script.getlogs.v4 ${RESOLVE_BASE_DIR}/scripts/
copy_object ${UNPACK_DIR}/scripts/script.start ${RESOLVE_BASE_DIR}/scripts/
copy_object ${UNPACK_DIR}/share/default-config.dat ${RESOLVE_BASE_DIR}/share/
copy_object ${UNPACK_DIR}/share/log-conf.xml ${RESOLVE_BASE_DIR}/share/
copy_object ${UNPACK_DIR}/share/default_cm_config.bin ${RESOLVE_BASE_DIR}/share/

# Create udev rules
create_directory ${DEB_DIR}/lib/udev/rules.d
cat > ${DEB_DIR}/lib/udev/rules.d/75-davincipanel.rules <<EOF
SUBSYSTEM=="usb", ENV{DEVTYPE}=="usb_device", ATTRS{idVendor}=="1edb", MODE="0666"
EOF
cat > ${DEB_DIR}/lib/udev/rules.d/75-sdx.rules <<EOF
SUBSYSTEM=="usb", ENV{DEVTYPE}=="usb_device", ATTRS{idVendor}=="096e", MODE="0666"
EOF

# Install panel API library
create_directory ${DEB_DIR}/usr/lib/
extract_tgz ${UNPACK_DIR}/share/panels/dvpanel-framework-linux-x86_64.tgz ${DEB_DIR}/usr/lib/ libDaVinciPanelAPI.so

# Create common data dir
create_directory ${DEB_DIR}/var/BlackmagicDesign/DaVinci\ Resolve

# Create desktop shortcut
create_directory ${DEB_DIR}/usr/share/applications/
cat > ${DEB_DIR}/usr/share/applications/${DEB_NAME}.desktop <<EOF
[Desktop Entry]
Version=1.0
Encoding=UTF-8
Exec=/opt/resolve/bin/resolve
Icon=/opt/resolve/graphics/DV_Resolve.png
MimeType=
Name=${RESOLVE_NAME}
Terminal=false
Type=Application
Categories=AudioVideo
EOF

# Create Debian package
create_directory ${DEB_DIR}/DEBIAN
cat > ${DEB_DIR}/DEBIAN/control <<EOF
Package: ${DEB_NAME}
Version: ${DEB_VERSION}
Architecture: amd64
Maintainer: Blackmagic Design
Description: ${RESOLVE_NAME} made from ${INSTALLER_ARCHIVE}
Conflicts: ${DEB_CONFLICTS}
EOF
cat > ${DEB_DIR}/DEBIAN/postinst <<EOF
#!/bin/sh
chmod -R a+rw /opt/resolve/easyDCP
chmod -R a+rw /opt/resolve/LUT
chmod -R a+rw /opt/resolve/.license
chmod -R a+rw /var/BlackmagicDesign/"DaVinci Resolve"
EOF
make_executable ${DEB_DIR}/DEBIAN/postinst
check_command fakeroot
check_command dpkg-deb
fakeroot dpkg-deb -b ${DEB_DIR} ${DEB_DIR}.deb
if [ $? -ne 0 ] ; then
    ERRORS=$((ERRORS+1))
fi

# Clean up
rm -rf ${DEB_DIR}
rm -rf ${UNPACK_DIR}

# Report
echo "[DONE] errors reported ${ERRORS}"

exit $ERRORS
